import java.awt.Color; 
import ij.*; 
import ij.gui.*; 
import ij.plugin.*; 
import ij.process.*; 
 
 
 
/** This plugin segments objects using Size Interval Precision
 *  "Global Gray-level Thresholding Based on Object Size", Cytometry Part A, 89:4, 2016, pp. 385–390.
 *  @author Petter Ranefall 
 *  @version 1.0 
 *  @date 2015-11-27
 *
 *  Input is an interval of expected object sizes. The algorithm computes the threshold level that gives the maximum precision 
 *  with regard to area of objects inside the size interval (tp), in relation to area of objects outside the size interval (fp).
 *  To handle clustered objects there is a flag IgnoreLargeObjects. If true only objects with sizes below the size interval
 *  will contribute to fp, and objects with sizes above the size interval will be ignored. The recommendation is in most cases
 *  to set this flag TRUE.
*/ 
 
public class SizeIntervalPrecision_ implements PlugIn { 
	protected ImageStack stack; 
	protected ImagePlus imp; 
    int[] parNode; 
    int[] imArray; 
	int maxP=0, minP=0; 
	int minSize = 0; 
	int maxSize = 0; 
	int[] areas;   
	double[] sumHistInside; 
	double[] sumHistOutside; 
	boolean plot = false; 
	boolean darkBkg = false; 
	boolean ignoreLargeObjects = false;
		 
	public void run(String arg)  
	{ 
		if (showDialog()) 
			exec(); 
	} 
	 
	public boolean showDialog()  
	{ 
		GenericDialog gd = new GenericDialog("Size Interval Precision"); 
		gd.addStringField("MinSize:", "100", 4); 
		gd.addStringField("MaxSize:", "2000", 4); 
		gd.addCheckbox("IgnoreLargeObjects", true);
		gd.addCheckbox("DarkBkg", true); 
		gd.addCheckbox("Plot", true); 
		gd.showDialog(); 
		if (gd.wasCanceled()) 
			return false; 
		minSize=Integer.valueOf(gd.getNextString()).intValue(); 
		maxSize=Integer.valueOf(gd.getNextString()).intValue(); 
		ignoreLargeObjects = gd.getNextBoolean();
		darkBkg = gd.getNextBoolean();  
		plot = gd.getNextBoolean(); 
		return true; 
	} 
	 
	public int swapValue(int v) 
	{ 
		if (darkBkg) 
			return v; 
		return minP + maxP - v; 
	} 
	 
	public double[] swapHist(double[] a, int l) 
	{ 
		if (darkBkg) 
			return a; 
		double[] newA = new double[l]; 
		for (int i = 0; i < l; i++) 
			newA[i] = a[l-1-i]; 
		return newA; 
	} 
 
	public void exec()  
	{ 
		if (IJ.versionLessThan("1.27t")) 
			return;   
		imp = WindowManager.getCurrentImage(); 
		stack=imp.getStack(); 
		int width=imp.getWidth(); 
		int height=imp.getHeight(); 
		int nSlices=imp.getStackSize(); 
		int nPixels=width*height*nSlices; 
		String imageName = imp.getTitle(); 
		 
		//Create nodes 
        parNode=new int[nPixels]; 
         
		int type = imp.getType(); 
		int mask=0xff; 
		int slice=1; 
		int x,y,z,x0,x2,y0,y2,z0,z2; 
		int rest; 
		int i,j,k, t; 
		int p; 
		int q; 
		imArray = new int[nPixels];  
		if (type == ImagePlus.GRAY16)  
		{ 
			mask=0xffff; 
			i=0; 
			//Find min, max. Copy to imArray 
			int intP; 
			short[] pixels = (short[])stack.getPixels(1); 
			intP=(int)(mask&pixels[0]); 
			maxP=minP=(int)(intP); 
			pixels = null; 
			for(slice=1; slice<=nSlices; slice++) 
			{ 
				pixels = (short[])stack.getPixels(slice); 
				j=0; 
				for(y=0; y<height; y++) 
				{ 
					for(x=0; x<width; x++) 
					{ 
						intP=(int)(mask&pixels[j]); 
						p = (int)(intP); 
						if(p<minP) 
							minP=p; 
						if(p>maxP) 
							maxP=p; 
						imArray[i]=p; 
						i++; 
						j++; 
					} 
				} 
				pixels = null; 
			} 
		}   
		else if (type == ImagePlus.GRAY8)  
		{ 
			mask=0xff; 
			i=0; 
			//Find min, max. Copy to imArray 
			byte[] pixels = (byte[])stack.getPixels(1); 
			maxP=minP=(int)(mask&pixels[0]); 
			pixels = null; 
			for(slice=1; slice<=nSlices; slice++) 
			{  
				pixels = (byte[])stack.getPixels(slice); 
				j=0; 
				for(y=0; y<height; y++) 
				{ 
					for(x=0; x<width; x++) 
					{ 
						p=(int)(mask&pixels[j]); 
						if(p<minP) 
							minP=p; 
						if(p>maxP) 
							maxP=p; 
						imArray[i]=p; 
						i++; 
						j++; 
					} 
				} 
				pixels = null; 
			} 
		} 
		else 
		{ 
			IJ.log("Pixel format not supported"); 
			return; 
		}
		if (!darkBkg) 
		{ 
			for (i=0; i<nPixels; i++) 
			{ 
				imArray[i] = (int)((int)minP + (int)maxP - (int)imArray[i]); 
			} 
		} 	 
		areas = new int[nPixels]; 
		for (i=0; i<nPixels; i++) 
		{ 
			areas[i]=1; 
		} 
		 
		//Sort points 
		// create a counting array, counts, with a member for  
		// each possible discrete value in the input.   
		// initialize all counts to 0. 
		int nLevels = maxP-minP + 1; 
		int[] counts = new int[nLevels]; 
		double[] grayHist = new double[nLevels];  
		sumHistInside = new double[nLevels]; 
		sumHistOutside = new double[nLevels]; 
		// for each value in the unsorted array, increment the 
		// count in the corresponding element of the count array 
		for (i=0; i<nPixels; i++) 
		{ 
			counts[imArray[i]-minP]++; 
		}  
		// accumulate the counts - the result is that counts will hold 
		// the offset into the sorted array for the value associated with that index 
		for (i=1; i<nLevels; i++) 
		{  
			counts[i] += counts[i-1]; 
		} 
		// store the elements in a new ordered array 
		int[] sortedIndex = new int[nPixels]; 
		for (i = nPixels-1; i >= 0; i--) 
		{ 
			// decrementing the counts value ensures duplicate values in A 
			// are stored at different indices in sorted. 
			sortedIndex[--counts[imArray[i]-minP]] = i;                 
		} 
 
		//Init nodes 
		for (i=0; i<nPixels; i++) 
		{ 
			parNode[i]=i; 
		} 
		//Search in decreasing order 
		int curNode; 
		int adjNode; 
		for (i = nPixels-1; i >= 0; i--) 
		{ 
			j=sortedIndex[i]; 
			curNode=j; 
			 
			z=j/(width*height); 
			rest=j-z*(width*height); 
			y=rest/width; 
			x=rest-y*width; 
			 
			z0=z-1; 
			z2=z+1; 
			y0=y-1; 
			y2=y+1; 
			x0=x-1; 
			x2=x+1; 
 
			//Later neigbours x2,y2,z2 
			boolean foundNeighbour = false; 
			if(z2<nSlices) 
			{ 
				k=x+width*(y+z2*height); 
				if(imArray[k]>=imArray[j]) 
				{ 
					adjNode=findNode(k); 
					if(curNode!=adjNode) 
					{ 
						curNode=mergeNodes(adjNode,curNode); 
					} 
				} 
			} 
			if(y2<height) 
			{ 
				k=x+width*(y2+z*height); 
				if(imArray[k]>=imArray[j]) 
				{ 
					adjNode=findNode(k); 
					if(curNode!=adjNode) 
					{ 
						curNode=mergeNodes(adjNode,curNode); 
					} 
				} 
			} 
			if(x2<width) 
			{ 
				k=x2+width*(y+z*height); 
				if(imArray[k]>=imArray[j]) 
				{ 
					adjNode=findNode(k); 
					if(curNode!=adjNode) 
					{ 
						curNode=mergeNodes(adjNode,curNode); 
					} 
				} 
			} 
			//Earlier neighbours x0,y0,z0. No need to check = 
			if(x0>=0) 
			{ 
				k=x0+width*(y+z*height); 
				if(imArray[k]>imArray[j]) 
				{ 
					adjNode=findNode(k); 
					if(curNode!=adjNode) 
					{ 
						curNode=mergeNodes(adjNode,curNode); 
					} 
					 
				} 
			} 
			if(y0>=0) 
			{ 
				k=x+width*(y0+z*height); 
				if(imArray[k]>imArray[j]) 
				{ 
					adjNode=findNode(k); 
					if(curNode!=adjNode) 
					{ 
						curNode=mergeNodes(adjNode,curNode); 
					} 
				} 
			} 
			if(z0>=0) 
			{ 
				k=x+width*(y+z0*height); 
				if(imArray[k]>imArray[j]) 
				{ 
					adjNode=findNode(k); 
					if(curNode!=adjNode) 
					{ 
						curNode=mergeNodes(adjNode,curNode); 
					} 
				} 
			} 
		} 
		double[] xValues = new double[nLevels]; 
		double[] yValuesPrecisionInsideOutside = new double[nLevels]; 
		double[] yValuesRecallInsideOutside = new double[nLevels];
		double maxSumHistInside = 0;
		for(i=minP; i<=maxP; i++) 
		{
			if (sumHistInside[i-minP] > maxSumHistInside && sumHistInside[i-minP] > sumHistOutside[i-minP])
			{
				maxSumHistInside = sumHistInside[i-minP];
			}
		}
		boolean maxFound = false; 
		int thr = minP;
		if (maxSumHistInside > 0)
		{
			for(i=minP; i<=maxP; i++) 
			{  
				if (sumHistInside[i-minP] > 0)
				{
					yValuesPrecisionInsideOutside[i-minP] = sumHistInside[i-minP] / (sumHistInside[i-minP] + sumHistOutside[i-minP]);
					yValuesRecallInsideOutside[i-minP] = sumHistInside[i-minP] / (maxSumHistInside);
				}
				else
				{
					yValuesPrecisionInsideOutside[i-minP] = 0;
					yValuesRecallInsideOutside[i-minP] = 0;
				}
				xValues[i-minP]= i;
			}  
			double maxPrecision = 0.5;;
			for(i=minP; i!=maxP; i++) 
			{ 
				if (yValuesPrecisionInsideOutside[i-minP] > maxPrecision && yValuesRecallInsideOutside[i-minP] > 0.5) 
				{ 
					maxPrecision = yValuesPrecisionInsideOutside[i-minP]; 
					maxFound = true; 
					thr = i; 
				}
			}
		}
		if (!maxFound)
		{ 
			thr = maxP + 1;
			IJ.log("No maximum precision found. Using thr: " + swapValue(thr)); 
		} 
		else
		{
			IJ.log("thr: " + thr);
		}
		 
		String xLabelText="Intensity (thr: "+swapValue(thr)+")"; 
		if (plot) 
		{ 
			PlotWindow plotWin = new PlotWindow("Size Interval Precision " + imageName,xLabelText,"Precision",xValues,swapHist(yValuesPrecisionInsideOutside, nLevels)); 
			plotWin.setLimits(minP, maxP, 0, 1);  
			plotWin.setColor(Color.red); 
			plotWin.draw(); 
		} 
		 
		i=0;  
		if (type == ImagePlus.GRAY16)  
		{ 
			for( slice=1; slice<=nSlices; slice++) 
			{ 
				short[] pixels = (short[])stack.getPixels(slice); 
				j=0; 
				for( y=0; y<height; y++) 
				{ 
					for( x=0; x<width; x++) 
					{ 
						q=imArray[i]; 
						if(q>=thr) 
							t=255; 
						else 
							t=0; 
						pixels[j]=(short)t; 
						i++; 
						j++; 
					} 
				} 
				pixels = null; 
			} 
		} 
		else if (type == ImagePlus.GRAY8)  
		{ 
			for( slice=1; slice<=nSlices; slice++) 
			{ 
				byte[] pixels = (byte[])stack.getPixels(slice); 
				j=0; 
				for( y=0; y<height; y++) 
				{ 
					for( x=0; x<width; x++) 
					{ 
						q=imArray[i]; 
						if(q>=thr) 
							t=255; 
						else 
							t=0; 
						pixels[j]=(byte)t; 
						i++; 
						j++; 
					} 
				} 
				pixels = null; 
			} 
		} 
		if(type!=ImagePlus.GRAY8) 
		{ 
			//Convert to 8-bit 
			ImageStack stack2 = new ImageStack(width, height); 
			String stackLabel; 
			ImageProcessor ip1, ip2; 
			boolean scale = false; 
			for( slice=1; slice<=nSlices; slice++)  
			{ 
				stackLabel = stack.getSliceLabel(1); 
				ip1 = stack.getProcessor(1); 
				ip2 = ip1.convertToByte(scale); 
				stack.deleteSlice(1); 
				stack2.addSlice(stackLabel, ip2); 
			} 
			imp.setStack(null, stack2); 
			stack2 = null; 
		} 
		imp.updateAndRepaintWindow(); 
	 
		//Deallocate arrays 
		parNode = null; 
		imArray = null; 
		areas = null;
		sumHistInside = null; 
		sumHistOutside = null; 
		counts = null;
		xValues = null; 
		yValuesPrecisionInsideOutside = null; 
		sortedIndex = null; 
		imp = null; 
		stack = null; 
		IJ.freeMemory(); 
	} 
	 
	public int findNode(int e) 
	{ 
		if(parNode[e]!=e) 
		{ 
			int root = findNode(parNode[e]); 
			parNode[e] = root; 
			return root; 
		} 
		else 
		{ 
			return e; 
		} 
	} 
	 
	public int mergeNodes(int e1,int e2) 
	{ 
		int res; 
		if(imArray[e1]==imArray[e2]) 
		{ 
			res=Math.max(e1,e2); 
			int m=Math.min(e1,e2); 
			areas[res] += areas[m]; 
			parNode[m]=res; 
		} 
		else 
		{ 
			parNode[e1]=e2; 
			areas[e2] += areas[e1]; 
			double size = areas[e1]; 
			if (size >= minSize && size <= maxSize) 
			{ 
				for(int k = imArray[e1]; k!=imArray[e2]; k--) 
				{ 
					sumHistInside[k-minP] += size; 
				} 
			} 
			else if (ignoreLargeObjects) 
			{ 
				if (size < minSize) 
				{ 
					for(int k = imArray[e1]; k!=imArray[e2]; k--) 
					{ 
						sumHistOutside[k-minP] += size; 
					} 
				} 
			} 
			else  
			{ 
				for(int k = imArray[e1]; k!=imArray[e2]; k--) 
				{ 
					sumHistOutside[k-minP] += size; 
				} 
			} 
				 
			res=e2; 
		} 
		return res; 
	} 
	 
	void error()  
	{ 
		IJ.showMessage("SizeIntervalPrecision", "Error"); 
	} 
} 
