"""<b>SizeIntervalPrecision</b> Object Feature based Gray-level Threshold using optimized Global Precision
<hr>
<b>SizeIntervalPrecision</b> produces a binary image
based on a threshold computed as the threshold level that optimized the Global Precision with regards to a given
expected object size range.

by Petter Ranefall 2015

Ranefall P, Wahlby C. "Global Gray-level Thresholding Based on Object Size", Cytometry Part A, 89:4, 2016, pp. 385-390.
"""

import _sizeIntervalPrecision as sip
import time
import numpy as np
import array
import math

import cellprofiler.module as cpm
import cellprofiler.setting as cps
import cellprofiler.image as cpi

import StringIO

class SizeIntervalPrecision(cpm.Module):
    variable_revision_number = 1
    module_name = "SizeIntervalPrecision"
    category = "Image Processing"
        
    def create_settings(self):
        #
        # The ImageNameSubscriber knows about the images that were provided
        # by all of the previous modules and will display those to the user
        # in a drop-down choice box.
        #
        self.input_image_name = cps.ImageNameSubscriber("Input image")
        
        self.diameter_interval = cps.IntegerRange(
            'Typical diameter of objects, in pixel units (Min, Max)', value=(75,150),minval=1, doc="""
            Typical diameter of objects, in pixel units (Min, Max))"""%globals())
        
        self.thr_interval = cps.FloatRange(
            'Lower and upper bounds of threshold', value=(0,1),minval=0, doc="""
            Lower and upper bounds of threshold"""%globals())
        
        self.ignore_large = cps.Binary(
            "Ignore large objects",
            True, doc='''
            Use this setting to ignore objects above the upper interval limit. This is used to handle clustered objects. The recommendation is that this should be set to Yes for most applications.
            ''')

        #
        # The ImageNameProvider tells CellProfiler that this module will
        # provide an image.
        #
        self.output_image_name = cps.ImageNameProvider("Output image", 
                                                       "Thr")
        #
        # If you have a image processing filter, there's a good chance that
        # there are some parameters such as the sigma of a Gaussian or
        # some other sort of scale. You can add those to create_settings
        # on the lines below.
        #
    def settings(self):
        #
        # Add your settings to the list below.
        #
        return [self.input_image_name, self.diameter_interval, self.thr_interval, self.ignore_large, self.output_image_name]
    
    def run(self, workspace):
        image_set = workspace.image_set
        #
        # Get your image from the image set using the ImageNameProvider
        #
        image = workspace.image_set.get_image(self.input_image_name.value)
        #
        # Get the pixel data from the image. I've chosen to make a copy
        # of the pixel data for safety's sake. It's easy to inadvertantly
        # change the input image's data and that will go against the
        # expectations of your users.
        #
        pixel_data = image.pixel_data.copy()
        
        start = time.clock()
        r_min = self.diameter_interval.min / 2.0
        r_max = self.diameter_interval.max / 2.0
        area_interval_min = (int)(math.pi * r_min * r_min)
        area_interval_max = (int)(math.pi * r_max * r_max)
        thr = sip._sizeIntervalPrecision(pixel_data, (area_interval_min, area_interval_max), (self.thr_interval.min, self.thr_interval.max), int(self.ignore_large.value))
        end = time.clock()
        elapsed = end - start
        print("thr: %s" % str(thr))
        print("ComputeThreshold: %s s" % str(elapsed))
        pixel_data = (pixel_data >= thr) & image.mask

        
        #
        # Make a cpi.Image using the transformed pixel data
        # put your image back in the image set.
        #
        output_image = cpi.Image(pixel_data)
        image_set.add(self.output_image_name.value, output_image)
        #
        # Store the input and output images in the workspace so that
        # they can be displayed later
        #
        if workspace.show_frame:
            workspace.display_data.input_image = image.pixel_data
            workspace.display_data.output_image = pixel_data

    #
    # The display interface is changing / has changed.
    # This is a recipe to make yours work with both
    #
    def display(self, workspace, figure=None):
        if figure is None:
            figure = workspace.create_or_find_figure(subplots=(2, 1))
        else:
            figure.set_subplots((2, 1))
        figure.subplot_imshow_grayscale(
            0, 0, workspace.display_data.input_image,
            title = self.input_image_name.value)
        figure.subplot_imshow_grayscale(
            1, 0, workspace.display_data.output_image,
            title = self.output_image_name.value, 
            sharexy = figure.subplot(0,0))        
        
    def is_interactive(self):
        return False
