import java.awt.Color;
import ij.*;
import ij.gui.*;
import ij.plugin.*;
import ij.process.*;



/** This plugin segments objects using local adaptive thresholding optimizing Ellipsefit
 *  Ranefall P, Sadanandan SK, Wählby C. "Fast Adaptive Local Thresholding Based on Ellipse Fit", (International Symposium on Biomedical Imaging (ISBI'16), Prague, Czech Republic, April 13-16, 2016)
 *  @author Petter Ranefall 
 *  @version 1.0 
 *  @date 2015-11-27
 *
 *  Input is an interval of expected object sizes. The algorithm computes the per object threshold level 
 *  that gives the maximum ellipse fit.
 *  Additional parameters are allowed intervals of major- and minor axes, the ratio between those, and lowest ellipse fit.
*/ 

public class PerObjectEllipsefit_ implements PlugIn {
	protected ImageStack stack;
	protected ImagePlus imp;
    int[] parNode;
    short[] imArray;
    byte[] outputArray;
    double[] ellipseFit;
	double[] majorAxis;
	double[] minorAxis;
	short maxP=0, minP=0;
	int minSize = 0;
	int maxSize = 0;
	int minMajorAxis = 0;
	int maxMajorAxis = 0;
	int minMinorAxis = 0;
	int maxMinorAxis = 0;
	double minMajorMinorRatio = 1.0;
	double maxMajorMinorRatio = 10.0;
	
	double ellipseThr = 0;
	int[] areas;
	double[] sumX;
	double[] sumX2;
	double[] sumY;
	double[] sumY2;
	double[] sumXY;
	short[] maxValInObj;
	int width;
	int height;
	int nSlices;
	boolean darkBkg = false;
	int minPeak;
	
	//CONSTANTS
	public static final byte UNDEFINED = (byte)0;
	public static final byte NOT_OBJECT = (byte)1;
	public static final byte MAYBE_NOT_OBJECT = (byte)2;
	public static final byte OBJECT = (byte)3;
	
	public void run(String arg) 
	{
		if (showDialog())
			exec();
	}
	
	public boolean showDialog() 
	{
		GenericDialog gd = new GenericDialog("Per Object Ellipse fit");
		gd.addStringField("MinSize:", "700", 4);
		gd.addStringField("MaxSize:", "4000", 4);
		gd.addStringField("EllipseThr:", "0.88", 4);
		gd.addStringField("MinMajorAxis:", "30", 4);
		gd.addStringField("MaxMajorAxis:", "200", 4);
		gd.addStringField("MinMinorAxis:", "15", 4);
		gd.addStringField("MaxMinorAxis:", "30", 4);
		gd.addStringField("MinMajorMinorRatio:", "1.5", 4);
		gd.addStringField("MaxMajorMinorRatio:", "10.0", 4);
		gd.addStringField("MinPeak:", "0", 4);
		gd.addCheckbox("DarkBkg", true);
		gd.showDialog();
		if (gd.wasCanceled())
			return false;
		minSize=Integer.valueOf(gd.getNextString()).intValue();
		maxSize=Integer.valueOf(gd.getNextString()).intValue();
		ellipseThr=Double.valueOf(gd.getNextString()).doubleValue();
		minMajorAxis=Integer.valueOf(gd.getNextString()).intValue();
		maxMajorAxis=Integer.valueOf(gd.getNextString()).intValue();
		minMinorAxis=Integer.valueOf(gd.getNextString()).intValue();
		maxMinorAxis=Integer.valueOf(gd.getNextString()).intValue();
		minMajorMinorRatio=Double.valueOf(gd.getNextString()).doubleValue();
		maxMajorMinorRatio=Double.valueOf(gd.getNextString()).doubleValue();
		minPeak=Integer.valueOf(gd.getNextString()).intValue();
		darkBkg = gd.getNextBoolean();
		return true;
	}

	public void exec() 
	{
		IJ.log("Start");
		long startTime0 = System.currentTimeMillis();
		if (IJ.versionLessThan("1.27t"))
			return;  
		imp = WindowManager.getCurrentImage();
		stack=imp.getStack();
		width=imp.getWidth();
		height=imp.getHeight();
		nSlices=imp.getStackSize();
		if (nSlices != 1)
			return; //Only for 2D
		int nPixels=width*height;
		String imageName = imp.getTitle();
		
		//Create nodes
        parNode=new int[nPixels];
        
		int type = imp.getType();
		int mask=0xff;
		int slice=1;
		int x,y,x0,x2,y0,y2;
		int rest;
		int i,j,k, t;
		short p;
		int q;
		imArray = new short[nPixels];
		outputArray = new byte[nPixels];
		ellipseFit = new double[nPixels];
		majorAxis = new double[nPixels];
		minorAxis = new double[nPixels];
		sumX = new double[nPixels];
		sumX2 = new double[nPixels];
		sumY = new double[nPixels];
		sumY2 = new double[nPixels];
		sumXY = new double[nPixels]; 
		maxValInObj = new short[nPixels];
		if (type == ImagePlus.GRAY16) 
		{
			mask=0xffff;
			//Find min, max. Copy to imArray
			short[] pixels = (short[])stack.getPixels(1);
			int intP = (int)(mask&pixels[0]);
			maxP=minP=(short)(intP/2);
			for (i=0; i<nPixels; i++)
			{
				intP=(int)(mask&pixels[i]);
				p = (short)(intP/2);
				if(p<minP)
					minP=p;
				if(p>maxP)
					maxP=p;
				imArray[i]=p;
			}
		}  
		else if (type == ImagePlus.GRAY8) 
		{
			mask=0xff;
			//Find min, max. Copy to imArray
			byte[] pixels = (byte[])stack.getPixels(1);
			p=(short)(mask&pixels[0]);
			maxP=minP=p;
			pixels = (byte[])stack.getPixels(1);
			for (i=0; i<nPixels; i++)
			{
				p=(short)(mask&pixels[i]);
				if(p<minP)
					minP=p;
				if(p>maxP)
					maxP=p;
				imArray[i]=p;
			}
		}
		else
		{
			IJ.log("Pixel format not supported");
			return;
		}
		if (!darkBkg)
		{
			for (i=0; i<nPixels; i++)
			{
				imArray[i] = (short)((int)minP + (int)maxP - (int)imArray[i]);
			}
		}
		areas = new int[nPixels];
		for (i=0; i<nPixels; i++)
		{
			areas[i]=1;
		}
		i=0;
		for(y=0; y<height; y++)
		{
			for(x=0; x<width; x++)
			{
				sumX[i] = x;
				sumX2[i] = x*x;
				sumY[i] = y;
				sumY2[i] = y*y;
				sumXY[i] = x*y;
				i++;
			}
		}
		
		//Sort points
		// create a counting array, counts, with a member for 
		// each possible discrete value in the input.  
		// initialize all counts to 0.
		int nLevels = maxP-minP + 1;
		int[] counts = new int[nLevels];
		// for each value in the unsorted array, increment the
		// count in the corresponding element of the count array
		for (i=0; i<nPixels; i++)
		{
			counts[imArray[i]-minP]++;
		}
		// accumulate the counts - the result is that counts will hold
		// the offset into the sorted array for the value associated with that index
		for (i=1; i<nLevels; i++)
		{
			counts[i] += counts[i-1];
		}
		// store the elements in a new ordered array
		int[] sortedIndex = new int[nPixels];
		for (i = nPixels-1; i >= 0; i--)
		{
			// decrementing the counts value ensures duplicate values in A
			// are stored at different indices in sorted.
			sortedIndex[--counts[imArray[i]-minP]] = i;                
		}

		//Init nodes
		for (i=0; i<nPixels; i++)
		{
			parNode[i]=i;
			maxValInObj[i] = imArray[i];
		}
		//Search in decreasing order
		int curNode;
		int adjNode;
		for (i = nPixels-1; i >= 0; i--)
		{
			j=sortedIndex[i];
			curNode=j;
			
			y=j/width;
			x=j-y*width;			
			
			y0=y-1;
			y2=y+1;
			x0=x-1;
			x2=x+1;

			//Later neigbours x2,y2
			if(y2<height)
			{
				k=x+width*y2;
				if(imArray[k]>=imArray[j])
				{
					adjNode=findNode(k);
					if(curNode!=adjNode)
					{
						curNode=mergeNodes(adjNode,curNode);
					}
				}
			}
			if(x2<width)
			{
				k=x2+width*y;
				if(imArray[k]>=imArray[j])
				{
					adjNode=findNode(k);
					if(curNode!=adjNode)
					{
						curNode=mergeNodes(adjNode,curNode);
					}
				}
			}
			//Earlier neighbours x0,y0. No need to check =
			if(x0>=0)
			{
				k=x0+width*y;
				if(imArray[k]>imArray[j])
				{
					adjNode=findNode(k);
					if(curNode!=adjNode)
					{
						curNode=mergeNodes(adjNode,curNode);
					}
					
				}
			}
			if(y0>=0)
			{
				k=x+width*y0;
				if(imArray[k]>imArray[j])
				{
					adjNode=findNode(k);
					if(curNode!=adjNode)
					{
						curNode=mergeNodes(adjNode,curNode);
					}
				}
			}
				
		}
		for (i = nPixels-1; i >= 0; i--)
		{
			j=sortedIndex[i];
			if (outputArray[j] == UNDEFINED)
			{
				int e = j;
				while(imArray[e] == imArray[parNode[e]] && outputArray[e] == UNDEFINED)
					e = parNode[e];
				if (outputArray[e] == UNDEFINED)
				{
					findBestEllipseLevel(j, minSize, maxSize, ellipseThr, minMajorAxis, maxMajorAxis, minMinorAxis, maxMinorAxis, minMajorMinorRatio, maxMajorMinorRatio, minPeak, -1);
				}
				else
				{
					int e1 = j;
					while(e1 != e)
					{
						outputArray[e1] = outputArray[e];
						e1 = parNode[e1];
					}
				}
			}
		}
		//Handle MAYBE
		for (i = nPixels-1; i >= 0; i--)
		{
			j=sortedIndex[i];
			if (outputArray[j] == MAYBE_NOT_OBJECT)
			{
				findBestEllipseLevel(j, minSize, maxSize, ellipseThr, minMajorAxis, maxMajorAxis, minMinorAxis, maxMinorAxis, minMajorMinorRatio, maxMajorMinorRatio, minPeak, -1);
			}
		}
		for (i = nPixels-1; i >= 0; i--)
		{
			j=sortedIndex[i];
			if (outputArray[j] == MAYBE_NOT_OBJECT)
			{
				findObjBelow(j, maxSize);
			}
		}
		
		if (type == ImagePlus.GRAY16) 
		{
			short[] pixels = (short[])stack.getPixels(1);

			for (i=0; i<nPixels; i++)
			{
				if(outputArray[i] == OBJECT)
					t=255;
				else
					t=0;
				pixels[i]=(short)t;
			}
		}
		else if (type == ImagePlus.GRAY8) 
		{
			byte[] pixels = (byte[])stack.getPixels(1);

			for (i=0; i<nPixels; i++)
			{
				if(outputArray[i] == OBJECT)
					t=255;
				else
					t=0;
				pixels[i]=(byte)t;				
			}
		}
				
				
		if(type!=ImagePlus.GRAY8)
		{
			//Convert to 8-bit
			ImageStack stack2 = new ImageStack(width, height);
			String stackLabel;
			ImageProcessor ip1, ip2;
			boolean scale = false;
			stackLabel = stack.getSliceLabel(1);
			ip1 = stack.getProcessor(1);
			ip2 = ip1.convertToByte(scale);
			stack.deleteSlice(1);
			stack2.addSlice(stackLabel, ip2);
			imp.setStack(null, stack2);
		}
		imp.updateAndRepaintWindow();
		
		//Deallocate arrays	
		parNode = null; 
		imArray = null; 
		areas = null;
		counts = null; 
		sortedIndex = null; 
		outputArray = null; 
		ellipseFit = null; 
		majorAxis = null; 
		minorAxis = null; 
		sumX = null; 
		sumX2 = null; 
		sumY = null; 
		sumY2 = null; 
		sumXY = null;  
		maxValInObj = null;
		imp = null; 
		stack = null; 
		IJ.freeMemory(); 
		IJ.log("Done");
		long stopTime = System.currentTimeMillis();
		IJ.log("TotalTime " + (stopTime - startTime0) + " ms");
	}
	
	public int findNode(int e)
	{
		if(parNode[e]!=e)
		{
			int root = findNode(parNode[e]);
			//parNode[e] = root; //This cannot be used here
			return root;
		}
		else
		{
			return e;
		}
	}
	
	public boolean findObjBelow(int e, double maxS)
	{
		int y=e/width;
		int x=e-y*width;
		if (parNode[e] == e)
		{
			outputArray[e] = NOT_OBJECT;			
			return false;
		}
		if (outputArray[parNode[e]] == OBJECT)
		{
			outputArray[e] = OBJECT;
			return true;
		}
		if (areas[parNode[e]] > maxS)
		{	
			outputArray[e] = NOT_OBJECT;			
			return false;
		}
		boolean found = findObjBelow(parNode[e], maxS);
		if (found)
			outputArray[e] = outputArray[parNode[e]];
		else
			outputArray[e] = NOT_OBJECT;
		return found;			
	}
	
	public int findBestEllipseLevel(int e, double minS, double maxS, double eThr, double minMajor, double maxMajor, double minMinor, double maxMinor, double minMajorMinor, double maxMajorMinor, int minPeak, int optE)
	{
		double optEf;
		double optArea;
		if (optE >= 0)
		{
			optEf = ellipseFit[optE];
			optArea = areas[optE];
		}
		else
		{
			optEf = 0;
			optArea = 0;
		}
		int startE = e;
		while(imArray[e] == imArray[parNode[e]] && parNode[e] != e)
			e = parNode[e];
		if (outputArray[e] == OBJECT)
		{
			return e;
		}
		if (outputArray[e] == NOT_OBJECT)
		{
			return optE;
		}
		if (parNode[e] == e)
		{
			outputArray[e] = NOT_OBJECT;
			return optE;
		}
		if (areas[e] > maxS)
		{
			outputArray[startE] = NOT_OBJECT;
			while(startE != parNode[startE] && outputArray[parNode[startE]] == UNDEFINED)
			{
				startE = parNode[startE];
				outputArray[startE] = NOT_OBJECT;
			}
			return optE;
		}
		double majorMinorRatio = majorAxis[e]/minorAxis[e];
		short diff = (short)(maxValInObj[e] - imArray[parNode[e]]);
		if (areas[e] >= minS && ellipseFit[e] > optEf && ellipseFit[e] > eThr && majorAxis[e] >= minMajor && majorAxis[e] <= maxMajor && minorAxis[e] >= minMinor && minorAxis[e] <= maxMinor && majorMinorRatio >= minMajorMinor && majorMinorRatio <= maxMajorMinor && diff >= minPeak)
		{			
			optEf = ellipseFit[e];
			optArea = areas[e];
			optE = e;
		}
		optE = findBestEllipseLevel(parNode[e], minS, maxS, eThr, minMajor, maxMajor, minMinor, maxMinor, minMajorMinor, maxMajorMinor, minPeak, optE);
		if (optE >= 0)
		{
			optEf = ellipseFit[optE];
			optArea = areas[optE];
		}
		else
		{
			optEf = 0;
			optArea = 0;
		}
		if (optE >= 0 && imArray[e]  >= imArray[optE]) //Opt found below
		{
			while(startE != e)
			{
				outputArray[startE] = OBJECT;
				startE = parNode[startE];
			}
			outputArray[e] = OBJECT;
			return optE;
		}
		
		// Below best level
		outputArray[startE] = MAYBE_NOT_OBJECT;
		if (optArea > areas[startE])
				areas[startE] = 0;
		else
			areas[startE] -= optArea;
		
		while(startE != e)
		{
			startE = parNode[startE];
			outputArray[startE] = MAYBE_NOT_OBJECT;			
			if (optArea > areas[startE])
				areas[startE] = 0;
			else
				areas[startE] -= optArea;
		}
		return optE;	
	}
	
	public int mergeNodes(int e1,int e2)
	{
		int res;
		int m;
		
		if(imArray[e1]==imArray[e2])
		{
			res=Math.max(e1,e2);
			m=Math.min(e1,e2);
		}
		else
		{
			res=e2;
			m=e1;
		}
		areas[res] += areas[m];
		parNode[m]=res;
		sumX[res] += sumX[m];
		sumX2[res] += sumX2[m];
		sumY[res] += sumY[m];
		sumY2[res] += sumY2[m];
		sumXY[res] += sumXY[m];
		if (maxValInObj[m] > maxValInObj[res])
			maxValInObj[res] = maxValInObj[m];
		if (areas[res] > 1)
		{
			double varX = (sumX2[res] - sumX[res]*sumX[res]/areas[res]);
			double varY = (sumY2[res] - sumY[res]*sumY[res]/areas[res]);
			double covXY = (sumXY[res] - sumX[res]*sumY[res]/areas[res]);
			double varXPlusVarY = varX+varY;
			double sqrtExpr = Math.sqrt(varXPlusVarY*varXPlusVarY-4*(varX*varY-covXY*covXY));
			if (varXPlusVarY > sqrtExpr)
			{
				double r1 = 2.0*Math.sqrt((varXPlusVarY+sqrtExpr)/(2*areas[res]));
				double r2 = 2.0*Math.sqrt((varXPlusVarY-sqrtExpr)/(2*areas[res]));
				double ellipseArea = Math.PI * r1 * r2;
				if (ellipseArea > 0)
				{
					ellipseFit[res] = areas[res]/ellipseArea;
					majorAxis[res] = 2.0 * r1;
					minorAxis[res] = 2.0 * r2;
				}
			}
		}
		return res;
	}
	
	void error() 
	{
		IJ.showMessage("PerObjectEllipsefit", "Error");
	}
}
