"""<b>perobjectellipsefit</b> Object Feature based per object Gray-level Threshold using ellipse fit
<hr>
<b>perobjectellipsefit</b> produces a binary image
based on a per object threshold computed as the threshold level that optimized ellipse fit.

by Petter Ranefall 2015

Ranefall P, Sadanandan SK, Wahlby C. "Fast Adaptive Local Thresholding Based on Ellipse Fit", (International Symposium on Biomedical Imaging (ISBI'16), Prague, Czech Republic, April 13-16, 2016)
"""

import _perobjectellipsefit as poe
import time
import numpy as np
import array
import math

import cellprofiler_core.module as cpm
import cellprofiler_core.setting as cps
import cellprofiler_core.image as cpi

class perobjectellipsefit(cpm.Module):
    variable_revision_number = 1
    module_name = "perobjectellipsefit"
    category = "Image Processing"
    
    def create_settings(self):
        #
        # The ImageNameSubscriber knows about the images that were provided
        # by all of the previous modules and will display those to the user
        # in a drop-down choice box.
        #
        self.input_image_name = cps.subscriber.ImageSubscriber("Input image")
        
        self.diameter_interval = cps.range.IntegerRange(
            'Typical diameter of objects, in pixel units (Min, Max)', value=(75,150),minval=1, doc="""
            Typical diameter of objects, in pixel units (Min, Max))"""%globals())
        
        self.ellipsefit_thr = cps.text.Float(
            "Lowest allowed ellipse fit",value=0,minval=0,maxval=1,doc = '''
            Use this setting to remove non-elliptic objects. The maximum value for a perfect ellipse is 1.0, between 0 and 1 otherwise.''')
        
        
        self.major_interval = cps.range.IntegerRange(
            'Range major axis', value=(30,200),minval=0, doc="""
            major axis interval (pixels)"""%globals())  
            
        self.minor_interval = cps.range.IntegerRange(
            'Range minor axis', value=(15,30),minval=0, doc="""
            minor axis interval (pixels)"""%globals())
        
        self.major_minor_ratio_interval = cps.range.FloatRange(
            'Range major/minor axis', value=(1,10),minval=1,doc="""
            Range major/minor axis interval"""%globals())
        
        self.thr_interval = cps.range.FloatRange(
            'Lower and upper bounds of threshold', value=(0,1),minval=0, maxval=1, doc="""
            Lower and upper bounds of threshold"""%globals())

        self.min_peak = cps.text.Float(
            "Lowest allowed intensity difference between min and max within an object",value=0.1,minval=0,maxval=1,doc = '''
            Lowest allowed intensity difference between min and max within an object.''')
            
        #
        # The ImageNameProvider tells CellProfiler that this module will
        # provide an image.
        #
        self.output_image_name = cps.text.ImageName("Output image",
                                                       "Thr")
        #
        # If you have a image processing filter, there's a good chance that
        # there are some parameters such as the sigma of a Gaussian or
        # some other sort of scale. You can add those to create_settings
        # on the lines below.
        #
    def settings(self):
        #
        # Add your settings to the list below.
        #
        return [self.input_image_name, self.diameter_interval, self.ellipsefit_thr, self.major_interval, self.minor_interval, self.major_minor_ratio_interval, self.thr_interval, self.min_peak, self.output_image_name]
       
    def run(self, workspace):
        image_set = workspace.image_set
        #
        # Get your image from the image set using the ImageNameProvider
        #
        image = workspace.image_set.get_image(self.input_image_name.value)
        #
        # Get the pixel data from the image. I've chosen to make a copy
        # of the pixel data for safety's sake. It's easy to inadvertantly
        # change the input image's data and that will go against the
        # expectations of your users.
        #
        pixel_data = image.pixel_data.copy()
        pixel_data = pixel_data.astype("double")  # This is new
        
        start = time.perf_counter()
        r_min = self.diameter_interval.min / 2.0
        r_max = self.diameter_interval.max / 2.0
        area_interval_min = (int)(math.pi * r_min * r_min)
        area_interval_max = (int)(math.pi * r_max * r_max)
        res = poe._perobjectellipsefit(pixel_data, (area_interval_min, area_interval_max), float(str(self.ellipsefit_thr)), (self.major_interval.min, self.major_interval.max), (self.minor_interval.min, self.minor_interval.max), (self.major_minor_ratio_interval.min, self.major_minor_ratio_interval.max), (self.thr_interval.min, self.thr_interval.max), float(str(self.min_peak)))    
        end = time.perf_counter()
        elapsed = end - start
        print("Compute ellipsefit: %s s" % str(elapsed))
        
        #
        # Make a cpi.Image using the transformed pixel data
        # put your image back in the image set.
        #
        res = (res > 0.5) & image.mask
        output_image = cpi.Image(res)
        image_set.add(self.output_image_name.value, output_image)
        #
        # Store the input and output images in the workspace so that
        # they can be displayed later
        #
        if workspace.show_frame:
            workspace.display_data.input_image = image.pixel_data
            workspace.display_data.output_image = res

    #
    # The display interface is changing / has changed.
    # This is a recipe to make yours work with both
    #
    def display(self, workspace, figure=None):
        if figure is None:
            figure = workspace.create_or_find_figure(subplots=(2, 1))
        else:
            figure.set_subplots((2, 1))
        figure.subplot_imshow_grayscale(
            0, 0, workspace.display_data.input_image,
            title = self.input_image_name.value)
        figure.subplot_imshow_grayscale(
            1, 0, workspace.display_data.output_image,
            title = self.output_image_name.value, 
            sharexy = figure.subplot(0,0))        
        
    def is_interactive(self):
        return False
