/* _perobjectellipsefit.c - C routines for perobjectellipsefit thresholding
 *
 * CellProfiler is distributed under the GNU General Public License,
 * but this file is licensed under the more permissive BSD license.
 * See the accompanying file LICENSE for details.
 *
 * Copyright (c) 2003-2009 Massachusetts Institute of Technology
 * Copyright (c) 2009-2015 Broad Institute
 * All rights reserved.
 *
 * Please see the AUTHORS file for credits.
 *
 * Website: http://www.cellprofiler.org
 *
 *
 * perobjectellipsefit - compute threshold
 *             the context of a structuring element.
 *             arg1 - the float64 ndarray image
 *             arg2 - size interval (a 2-tuple)
 *             arg3 - threshold constraint (a 2-tuple)
 */
#include <stdlib.h>
#include <math.h>
#include <Python.h>
#include <numpy/arrayobject.h>

#ifndef min
#define min(a,b)            (((a) < (b)) ? (a) : (b))
#endif

#ifndef max
#define max(a,b)            (((a) > (b)) ? (a) : (b))
#endif

static void to_stdout(const char *text)
{
     PySys_WriteStdout("%s", text);     
}

struct pair {
    int value;
    long index;
};

static const int UNDEFINED = 0;
static const int NOT_OBJECT = 1;
static const int MAYBE_NOT_OBJECT = 2;
static const int OBJECT = 3;

#define PI 3.14159265358979323846

long my_pair_compare(const void *const first, const void *const second)
{
    const struct pair* a;
	const struct pair* b;
	a = (const struct pair*)first;
    b = (const struct pair*)second;
    if (a->value > b->value)
       return 1;
    if (a->value < b->value)
        return -1;
    if (a->index > b->index)
       return 1;
    return -1;
}

int sortIndices(long count, int values[], long sortedIndices[])
{
    struct pair* ab;
	long i;
	ab = (struct pair *) malloc(sizeof(struct pair)*count);
	
	if (!ab)
	{
		return 0;
	}
	
    for (i = 0; i<count; ++i) {
        ab[i].value = values[i];
        ab[i].index = i;
    }
    qsort(ab, count, sizeof(*ab), my_pair_compare);
    for (i=0; i<count; ++i){
        sortedIndices[i] = ab[i].index;
    }
	free(ab);
	return 1;
}

long _findNode(long n, long parentNodes[])
{
	long root;
    if (parentNodes[n] != n)
	{
        root = _findNode(parentNodes[n], parentNodes);
        return root;
	}
    return n;
}

long _mergeNodes(long n1, long n2, long areaLowerBound, long areaUpperBound, long lenData, 
				int intData[], long parentNodes[], long areas[], double ellipsefit[], double majorAxis[], double minorAxis[],
				double sumX[], double sumX2[], double sumY[], double sumY2[], double sumXY[], double maxValInObj[], int outputArray[])
{
	long par;
	long child;
    double varX;
    double varY;
    double covXY;
    double varXPlusVarY;
    double sqrtExpr;
    double r1;
    double r2;
    double ellipseArea;
	
    if (intData[n1] == intData[n2])
	{
        par = max(n1, n2);
        child = min(n1, n2);
	}
    else // pixels[n1] is always > pixels[n2] here
	{
        par = n2;
        child = n1;
	}
    areas[par] += areas[child];
    parentNodes[child] = par;
    sumX[par] += sumX[child];
    sumX2[par] += sumX2[child];
    sumY[par] += sumY[child];
    sumY2[par] += sumY2[child];
    sumXY[par] += sumXY[child];
	if (maxValInObj[child] > maxValInObj[par])
		maxValInObj[par] = maxValInObj[child];
    
    if (areas[par] > 1)
	{
        varX = (sumX2[par] - sumX[par]*sumX[par]/areas[par]);
        varY = (sumY2[par] - sumY[par]*sumY[par]/areas[par]);
        covXY = (sumXY[par] - sumX[par]*sumY[par]/areas[par]);
        varXPlusVarY = varX + varY;
        sqrtExpr = sqrt(varXPlusVarY*varXPlusVarY-4*(varX*varY-covXY*covXY));
        if (varXPlusVarY > sqrtExpr)
		{
            r1 = 2.0*sqrt((varXPlusVarY+sqrtExpr)/(2.0 * areas[par]));
            r2 = 2.0*sqrt((varXPlusVarY-sqrtExpr)/(2.0 * areas[par]));
            ellipseArea = PI * r1 * r2;
            if (ellipseArea > 0)
			{
                ellipsefit[par] = areas[par]/ellipseArea;
                majorAxis[par] = 2.0 * r1;
                minorAxis[par] = 2.0 * r2;
			}
		}
	}
    return par;
}

void _findBestEllipseLevel(int inputE, int areaLowerBound, int areaUpperBound, double ellipsefitThr, double minMajorAxis, double maxMajorAxis, double minMinorAxis, double maxMinorAxis, double minMajorMinorRatio, double maxMajorMinorRatio,
						   int intData[], long parentNodes[], long areas[], double ellipsefit[], double majorAxis[], double minorAxis[], int outputArray[], double thrLowerBound, double thrUpperBound, double minPeak, double maxValInObj[], double imdata[])
{    
    int e = inputE;
    int optE = -1;
    double optEf = 0;
    int optArea = 0;
    int startE = e;
    double majorMinorRatio = 1.0;
    int firstE = e;
	double diff = 0.0;
    while (1)
	{
        firstE = e;
        while (intData[e] == intData[parentNodes[e]] && parentNodes[e] != e)
		{
            e = parentNodes[e];
		}
        if (outputArray[e] == OBJECT)
		{
            while (startE != e)
			{
                outputArray[startE] = OBJECT;
                startE = parentNodes[startE];
			}
            return;
		}
        if (outputArray[e] == NOT_OBJECT)
		{
            if (optE >= 0)
			{
                while (startE != optE)
				{
                    outputArray[startE] = OBJECT;
                    startE = parentNodes[startE];
				}
                outputArray[optE] = OBJECT; 
                startE = parentNodes[optE];
			}
            while (startE != e)
			{
                outputArray[startE] = NOT_OBJECT;
                startE = parentNodes[startE];
			}
            return;
		}
        if (parentNodes[e] == e) // root
		{
            if (optE >= 0)
			{
                while (startE != optE)
				{
                    outputArray[startE] = OBJECT;
                    startE = parentNodes[startE];
				}
                outputArray[optE] = OBJECT;
                startE = parentNodes[optE];
			}
            while (startE != e)
			{
                outputArray[startE] = NOT_OBJECT;
                startE = parentNodes[startE];
			}
            return;
		}
        if (areas[e] > areaUpperBound || imdata[e] < thrLowerBound)
		{
            // set from firstE
            if (optE >= 0)
			{
                while (startE != optE)
				{
                    outputArray[startE] = OBJECT;
                    startE = parentNodes[startE];
				}
                outputArray[optE] = OBJECT;
                startE = parentNodes[optE];
                while (startE != e)
				{
                    outputArray[startE] = MAYBE_NOT_OBJECT;
					if (optArea > areas[startE])
						areas[startE] = 0;
					else
						areas[startE] -= optArea;
                    startE = parentNodes[startE];
				}
			}
            while (firstE != parentNodes[firstE] && outputArray[parentNodes[firstE]] == UNDEFINED)
			{
                firstE = parentNodes[firstE];
                outputArray[firstE] = NOT_OBJECT;
			}
            outputArray[firstE] = NOT_OBJECT;
            return;
		}
        majorMinorRatio = 1.0;
        if (minorAxis[e] > 0)
		{
            majorMinorRatio = majorAxis[e]/minorAxis[e];
		}
		diff = maxValInObj[e] - imdata[parentNodes[e]];
        if (areas[e] >= areaLowerBound && imdata[e] <= thrUpperBound && ellipsefit[e] > optEf && ellipsefit[e] > ellipsefitThr && majorAxis[e] >= minMajorAxis && majorAxis[e] <= maxMajorAxis && minorAxis[e] >= minMinorAxis && minorAxis[e] <= maxMinorAxis && majorMinorRatio >= minMajorMinorRatio && majorMinorRatio <= maxMajorMinorRatio && diff >= minPeak)
		{
            optEf = ellipsefit[e];
            optArea = areas[e];
            optE = e;
		}			
            
        e = parentNodes[e];
	}
}	
    
void _findObjBelow(int startE, int width, int height, int areaUpperBound, 
				   long parentNodes[], long areas[], int outputArray[], double thrLowerBound, double thrUpperBound, double imdata[])
{
    int e = startE;
    int res = NOT_OBJECT;
	int j = startE;
    while (1)
	{
        if (parentNodes[e] == e)
		{
            res = NOT_OBJECT;
            break;
		}
        if (outputArray[parentNodes[e]] == OBJECT)
		{
            res = OBJECT;
            break;
		}
        if (areas[parentNodes[e]] > areaUpperBound || imdata[e] < thrLowerBound)
		{
            res = NOT_OBJECT;
            break;
		}
        e = parentNodes[e];
	}
    while (j != e)
	{
        outputArray[j] = res;
        j = parentNodes[j];
	}
}
		
static PyObject *
_perobjectellipsefit(PyObject *self, PyObject *args)
{
	PyObject *image     = NULL; /* the image ndarray */
	PyObject *cimage    = NULL; /* a contiguous version of the array */
	PyObject *im_shape  = NULL; /* the image shape sequence */
    PyObject *sizeInterval = NULL; /* 2-tuple giving the size interval */
    PyObject *ellipsefitThrObj = NULL; /* double giving the ellipse fit threshold */
    PyObject *majorInterval = NULL; /* 2-tuple giving the major interval */
    PyObject *minorInterval = NULL; /* 2-tuple giving the major interval */
    PyObject *majorMinorRatioInterval = NULL; /* 2-tuple giving the major/minor ratio interval */
    PyObject *thrInterval = NULL; /* 2-tuple giving the threshold interval */
    PyObject *minPeakObj = NULL; /* double giving the minimum peak, i.e. difference between min and max within an object */
	long     height     = 0;    /* the image height */
	long     width      = 0;    /* the image width */
	double   *imdata    = NULL; /* the image data */
    PyObject *output    = NULL; /* the output ndarray */
    double   *out_data  = NULL; /* the output data */
	const char *error   = NULL;
	PyObject **shapes[] = { &im_shape, &im_shape };
	long     *slots[]   = { &height, &width };
    long      indices[]  = { 0,1 };
	long      i          = 0;
	long      j          = 0;
	long      k          = 0;
	npy_intp dims[2];
	long lenData;
	long *sortedIndices = NULL;
	int *intData = NULL;
	// long curLevel;
	
    long *parentNodes = NULL;
    long *areas = NULL;
    double *ellipsefit = NULL;
    double *majorAxis = NULL;
    double *minorAxis = NULL;
    double *sumX = NULL;
    double *sumX2 = NULL;
    double *sumY = NULL;
    double *sumY2 = NULL;
    double *sumXY = NULL;
	double *maxValInObj = NULL;
    int *outputArray = NULL;
	double minv;
	double maxv;
	double f;
	
	long x;
	long y;
	long prevX;
	long prevY;
	long nextX;
	long nextY;
	long curNode;
	long adjNode;
	long areaLowerBound;
	long areaUpperBound;
	double thrLowerBound;
	double thrUpperBound;
	long e;
	long e1;
	double ellipsefitThr = 0.88;
	long minMajorAxis = 30;
	long maxMajorAxis = 200;
	long minMinorAxis = 15;
	long maxMinorAxis = 30;
	double minMajorMinorRatio = 1.0;
	double maxMajorMinorRatio = 10.0;
	double minPeak = 0.1;
	printf("Start");
	import_array();
	image = PySequence_GetItem(args,0);
	if (! image) {
	  error = "Failed to get image from arguments";
	  goto exit;
	}
	cimage = image;//PyArray_ContiguousFromAny(image, NPY_DOUBLE,2,2);
	if (! cimage) {
	  error = "Failed to make a contiguous array from first argument";
	  goto exit;
	}
	printf("cimage done");
	im_shape = PyObject_GetAttrString(cimage,"shape");
	if (! im_shape) {
	  error = "Failed to get image.shape";
	  goto exit;
	}
	if (! PyTuple_Check(im_shape)) {
	  error = "image.shape not a tuple";
	  goto exit;
	}
	if (PyTuple_Size(im_shape) != 2) {
	  error = "image is not 2D";
	  goto exit;
	}
	for (i=0;i<2;i++) {
		PyObject *obDim = PyTuple_GetItem(*shapes[i],indices[i]);
		*(slots[i]) = PyLong_AsLong(obDim);
		if (PyErr_Occurred()) {
			error = "Array shape is not a tuple of integers";
			goto exit;
		}
	}
	
	dims[0] = height;
	dims[1] = width;	
	
	sizeInterval = PySequence_GetItem(args,1);
	if (! sizeInterval) 
	{
		  error = "Failed to get sizeInterval into structure from args";
		  goto exit;
	}
	if (! PyTuple_Check(sizeInterval)) 
	{
		  error = "sizeInterval is not a tuple";
		  goto exit;
	} 
	else 
	{
		PyObject *temp = PyTuple_GetItem(sizeInterval,0);
		if (! temp) 
		{
		   error = "Failed to get areaLowerBound from tuple";
		   goto exit;
		}
		areaLowerBound = PyLong_AsLong(temp);
		if (PyErr_Occurred()) 
		{
		   error = "areaLowerBound is not an integer";
		   goto exit;
		}
		temp = PyTuple_GetItem(sizeInterval,1);
		if (! temp) 
		{
		   error = "Failed to get areaUpperBound from tuple";
		   goto exit;
		}
		areaUpperBound = PyLong_AsLong(temp);
		if (PyErr_Occurred()) 
		{
		   error = "areaUpperBound is not an integer";
		   goto exit;
		}
	}
	
	ellipsefitThrObj = PySequence_GetItem(args,2);
	if (! ellipsefitThrObj) 
	{
		  error = "Failed to get ellipsefitThrObj into structure from args";
		  goto exit;
	}
	ellipsefitThr = PyFloat_AsDouble(ellipsefitThrObj);
	if (PyErr_Occurred()) 
	{
	   error = "ellipsefitThr is not a number";
	   goto exit;
	}
	
	majorInterval = PySequence_GetItem(args,3);
	if (! majorInterval) 
	{
		  error = "Failed to get majorInterval into structure from args";
		  goto exit;
	}
	if (! PyTuple_Check(majorInterval)) 
	{
		  error = "majorInterval is not a tuple";
		  goto exit;
	} 
	else 
	{
		PyObject *temp = PyTuple_GetItem(majorInterval,0);
		if (! temp) 
		{
		   error = "Failed to get minMajorAxis from tuple";
		   goto exit;
		}
		minMajorAxis = PyLong_AsLong(temp);
		if (PyErr_Occurred()) 
		{
		   error = "minMajorAxis is not a double";
		   goto exit;
		}
		temp = PyTuple_GetItem(majorInterval,1);
		if (! temp) 
		{
		   error = "Failed to get maxMajorAxis from tuple";
		   goto exit;
		}
		maxMajorAxis = PyLong_AsLong(temp);
		if (PyErr_Occurred()) 
		{
		   error = "maxMajorAxis is not an integer";
		   goto exit;
		}
	}
	
	minorInterval = PySequence_GetItem(args,4);
	if (! minorInterval) 
	{
		  error = "Failed to get minorInterval into structure from args";
		  goto exit;
	}
	if (! PyTuple_Check(minorInterval)) 
	{
		  error = "minorInterval is not a tuple";
		  goto exit;
	} 
	else 
	{
		PyObject *temp = PyTuple_GetItem(minorInterval,0);
		if (! temp) 
		{
		   error = "Failed to get minMinorAxis from tuple";
		   goto exit;
		}
		minMinorAxis = PyLong_AsLong(temp);
		if (PyErr_Occurred()) 
		{
		   error = "minMinorAxis is not a double";
		   goto exit;
		}
		temp = PyTuple_GetItem(minorInterval,1);
		if (! temp) 
		{
		   error = "Failed to get maxMinorAxis from tuple";
		   goto exit;
		}
		maxMinorAxis = PyLong_AsLong(temp);
		if (PyErr_Occurred()) 
		{
		   error = "maxMinorAxis is not an integer";
		   goto exit;
		}
	}
	
	majorMinorRatioInterval = PySequence_GetItem(args,5);
	if (! majorMinorRatioInterval) 
	{
		  error = "Failed to get majorMinorRatioInterval into structure from args";
		  goto exit;
	}
	if (! PyTuple_Check(majorMinorRatioInterval)) 
	{
		  error = "majorMinorRatioInterval is not a tuple";
		  goto exit;
	} 
	else 
	{
		PyObject *temp = PyTuple_GetItem(majorMinorRatioInterval,0);
		if (! temp) 
		{
		   error = "Failed to get minMajorMinorRatio from tuple";
		   goto exit;
		}
		minMajorMinorRatio = PyFloat_AsDouble(temp);
		if (PyErr_Occurred()) 
		{
		   error = "minMajorMinorRatio is not a double";
		   goto exit;
		}
		temp = PyTuple_GetItem(majorMinorRatioInterval,1);
		if (! temp) 
		{
		   error = "Failed to get maxMajorMinorRatio from tuple";
		   goto exit;
		}
		maxMajorMinorRatio = PyFloat_AsDouble(temp);
		if (PyErr_Occurred()) 
		{
		   error = "maxMajorMinorRatio is not an integer";
		   goto exit;
		}
	}
	
	thrInterval = PySequence_GetItem(args,6);
	if (! thrInterval) 
	{
		  error = "Failed to get thrInterval into structure from args";
		  goto exit;
	}
	if (! PyTuple_Check(thrInterval)) 
	{
		  error = "thrInterval is not a tuple";
		  goto exit;
	} 
	else 
	{
		PyObject *temp = PyTuple_GetItem(thrInterval,0);
		if (! temp) 
		{
		   error = "Failed to get thrLowerBound from tuple";
		   goto exit;
		}
		thrLowerBound = PyFloat_AsDouble(temp);
		if (PyErr_Occurred()) 
		{
		   error = "thrLowerBound is not a double";
		   goto exit;
		}
		temp = PyTuple_GetItem(thrInterval,1);
		if (! temp) 
		{
		   error = "Failed to get thrUpperBound from tuple";
		   goto exit;
		}
		thrUpperBound = PyFloat_AsDouble(temp);
		if (PyErr_Occurred()) 
		{
		   error = "thrUpperBound is not an integer";
		   goto exit;
		}
	}
	if (thrLowerBound > 1.0 || thrUpperBound < 0.0 || thrLowerBound > thrUpperBound)
	{
		error = "Error in threshold interval";
		goto exit;
	}
	
	minPeakObj = PySequence_GetItem(args,7);
	if (! minPeakObj) 
	{
		  error = "Failed to get minPeakObj into structure from args";
		  goto exit;
	}
	minPeak = PyFloat_AsDouble(minPeakObj);
	if (PyErr_Occurred()) 
	{
	   error = "minPeak is not a number";
	   goto exit;
	} 
	
	lenData = width * height;
	
	imdata = (double *)PyArray_DATA(cimage);
	if (! imdata) {
		error = "Failed to get image data";
		goto exit;
	}
	
	minv = imdata[0];
	maxv = imdata[0];
    for(i = 1; i < lenData; ++i)
	{
        if (imdata[i] < minv)
			minv = imdata[i];
		else if (imdata[i] > maxv)
			maxv = imdata[i];
	}
	if (minv == maxv) {
		error = "Image constant";
		goto exit;
	}
	f = 255.999 / (maxv-minv);
	intData = (int *) malloc(sizeof(int)*lenData);
	if (! intData) {
		error = "Failed to allocate memory";
		goto exit;
	}
    for(i = 0; i < lenData; ++i)
	{
        intData[i] = (int)(f * (imdata[i]-minv));
	}
	
	sortedIndices = (long *) malloc(sizeof(long)*lenData);
	if (! sortedIndices) {
		error = "Failed to allocate memory";
		goto exit;
	}
	
	if (!sortIndices(lenData, intData, sortedIndices))
	{
		error = "Failed to allocate memory";
		goto exit;
	}
	
    parentNodes = (long *) malloc(sizeof(long)*lenData); 
    areas = (long *) malloc(sizeof(long)*lenData);
	ellipsefit = (double *) calloc(lenData, sizeof(double));
	majorAxis = (double *) calloc(lenData, sizeof(double));
	minorAxis = (double *) calloc(lenData, sizeof(double));
	sumX = (double *) malloc(lenData*sizeof(double));
	sumX2 = (double *) malloc(lenData*sizeof(double));
	sumY = (double *) malloc(lenData*sizeof(double));
	sumY2 = (double *) malloc(lenData*sizeof(double));
	sumXY = (double *) malloc(lenData*sizeof(double));
	maxValInObj = (double *) malloc(lenData*sizeof(double));
	outputArray = (int *) calloc(lenData, sizeof(int));
	if (!(parentNodes && areas && ellipsefit && majorAxis && minorAxis &&
		sumX && sumX2 && sumY && sumY2 && sumXY && maxValInObj && outputArray))
	{
		error = "Failed to allocate memory";
		goto exit;
	}

    for(i = 0; i < lenData; ++i)
	{
		parentNodes[i] = i;
		areas[i] = 1;
		maxValInObj[i] = imdata[i];
	}
	i = 0;
	for (y = 0; y < height; ++y)
	{
		for (x = 0; x < width; ++x)
		{
			sumX[i] = x;
			sumX2[i] = x * x;
			sumY[i] = y;
			sumY2[i] = y * y;
			sumXY[i] = x * y;
			++i;
		}
	}
	
    for(i = lenData-1; i >= 0; i--)
	{
        j = sortedIndices[i];
        curNode = j;
        y = j/width;
        x = j-y*width;
        prevY = y-1;
        nextY = y+1;
        prevX = x-1;
        nextX = x+1;

        if (nextY<height)
		{
            k = x+width*nextY;
            if (intData[k] >= intData[j])
			{
                adjNode = _findNode(k, parentNodes);
                if (curNode != adjNode)
                    curNode = _mergeNodes(adjNode, curNode, areaLowerBound, areaUpperBound, lenData, 
										  intData, parentNodes, areas, ellipsefit, majorAxis, minorAxis,
										  sumX, sumX2, sumY, sumY2, sumXY, maxValInObj, outputArray);
			}
		}
        if (nextX<width)
		{
            k = nextX+width*y;
            if (intData[k] >= intData[j])
			{
                adjNode = _findNode(k, parentNodes);
                if (curNode != adjNode)
                    curNode = _mergeNodes(adjNode, curNode, areaLowerBound, areaUpperBound, lenData, 
										  intData, parentNodes, areas, ellipsefit, majorAxis, minorAxis,
										  sumX, sumX2, sumY, sumY2, sumXY, maxValInObj, outputArray);
			}
		}
        if (prevX >= 0)
		{
            k = prevX+width*y;
            if (intData[k] > intData[j])
			{
                adjNode = _findNode(k, parentNodes);
                if (curNode != adjNode)
                    curNode = _mergeNodes(adjNode, curNode, areaLowerBound, areaUpperBound, lenData, 
										  intData, parentNodes, areas, ellipsefit, majorAxis, minorAxis,
										  sumX, sumX2, sumY, sumY2, sumXY, maxValInObj, outputArray);
			}
		}
        if (prevY >= 0)
		{
            k = x+width*prevY;
            if (intData[k] > intData[j])
			{
                adjNode = _findNode(k, parentNodes);
                if (curNode != adjNode)
                    curNode = _mergeNodes(adjNode, curNode, areaLowerBound, areaUpperBound, lenData, 
										  intData, parentNodes, areas, ellipsefit, majorAxis, minorAxis,
										  sumX, sumX2, sumY, sumY2, sumXY, maxValInObj, outputArray);
			}
		}
    }
	
	// Handle UNDEFINED
    for(i = lenData-1; i >= 0; i--)
	{
        j = sortedIndices[i];
        if (outputArray[j] == UNDEFINED)
		{
            e = j;
            while (intData[e] == intData[parentNodes[e]] && outputArray[e] == UNDEFINED && e != parentNodes[e])
			{
                e = parentNodes[e];
			}
            if (outputArray[e] == UNDEFINED)
			{
                _findBestEllipseLevel(j, areaLowerBound, areaUpperBound, ellipsefitThr, minMajorAxis, maxMajorAxis, minMinorAxis, maxMinorAxis, minMajorMinorRatio, maxMajorMinorRatio,
				                      intData, parentNodes, areas, ellipsefit, majorAxis, minorAxis, outputArray, thrLowerBound, thrUpperBound, minPeak, maxValInObj, imdata);
			}
            else
			{
                e1 = j;
                while (e1 != e)
				{
                    outputArray[e1] = outputArray[e];
                    e1 = parentNodes[e1];
				}
			}
		}
	}
	
    // Handle MAYBE
    for(i = lenData-1; i >= 0; i--)
	{
        j = sortedIndices[i];
        if (outputArray[j] == MAYBE_NOT_OBJECT)
		{
			_findBestEllipseLevel(j, areaLowerBound, areaUpperBound, ellipsefitThr, minMajorAxis, maxMajorAxis, minMinorAxis, maxMinorAxis, minMajorMinorRatio, maxMajorMinorRatio,
								  intData, parentNodes, areas, ellipsefit, majorAxis, minorAxis, outputArray, thrLowerBound, thrUpperBound, minPeak, maxValInObj, imdata);
		}
	}
    
    // Handle MAYBE
    for(i = lenData-1; i >= 0; i--)
	{
        j = sortedIndices[i];
        if (outputArray[j] == MAYBE_NOT_OBJECT)
		{
            _findObjBelow(j, width, height, areaUpperBound, parentNodes, areas, outputArray, thrLowerBound, thrUpperBound, imdata);
		}
	}
	output = PyArray_SimpleNew(2, dims, NPY_DOUBLE);
	if (! output) {
		  error = "Failed to create output array";
		  goto exit;
	}
	out_data = (double *)PyArray_DATA(output);

	//Set result
	for (i = 0; i < lenData; ++i)
	{
		if (outputArray[i] == OBJECT)
            *out_data++ = 1.0;
		else
            *out_data++ = 0.0;
	}

exit:
    printf("exit");
	/*if (image) {
		Py_DECREF(image);
	}
	if (cimage) {
		Py_DECREF(cimage);
	}*/
	if (im_shape) {
		Py_DECREF(im_shape);
	}
	if (sizeInterval) {
		Py_DECREF(sizeInterval);
	}
	if (ellipsefitThrObj) {
		Py_DECREF(ellipsefitThrObj);
	}
	if (majorInterval) {
		Py_DECREF(majorInterval);
	}
	if (minorInterval) {
		Py_DECREF(minorInterval);
	}
	if (majorMinorRatioInterval) {
		Py_DECREF(majorMinorRatioInterval);
	}
	if (thrInterval) {
		Py_DECREF(thrInterval);
	}
	if (minPeakObj) {
		Py_DECREF(minPeakObj);
	}
	    
	free(parentNodes);
    free(areas);
    free(intData);
    free(ellipsefit);
    free(majorAxis);
    free(minorAxis);
    free(sumX);
    free(sumX2);
    free(sumY);
    free(sumY2);
    free(sumXY);
    free(outputArray);
	free(maxValInObj);

	if (error) {
		to_stdout(error);
	}
	printf("done");
    return output;
}

static PyMethodDef methods[] = {
    {"_perobjectellipsefit", (PyCFunction)_perobjectellipsefit, METH_VARARGS, NULL},
    { NULL, NULL, 0, NULL}
};

static struct PyModuleDef poemodule = {
    PyModuleDef_HEAD_INIT,
    "_perobjectellipsefit", /* module name */
    NULL, /* module documentation, may be NULL */
    -1,
    methods /* the methods array */
};


PyMODINIT_FUNC PyInit__perobjectellipsefit(void)
{
    return PyModule_Create(&poemodule);
}

