import java.awt.Color;
import ij.*;
import ij.gui.*;
import ij.plugin.*;
import ij.process.*;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.io.File;



/** This plugin segments objects using local adaptive thresholding optimizing Ellipsefit
 *  "Fast Adaptive Local Thresholding Based on Ellipse Fit"
 *  @author Petter Ranefall 
 *  @version 1.0 
 *  @date 2015-11-27
 *
 *  Input is an interval of expected object sizes. The algorithm computes the per object threshold level 
 *  that gives the maximum ellipse fit.
 *  Additional parameters are allowed intervals of major- and minor axes, the ratio between those, and lowest ellipse fit.
*/ 

public class PerObjectEllipsefit3D_ implements PlugIn {
	protected ImageStack stack;
	protected ImagePlus imp;
    int[] parNode;
    short[] imArray;
    byte[] outputArray;
    double[] ellipseFit;
	double[] majorAxis;
	double[] minorAxis;
	short maxP=0, minP=0;
	int minSize = 0;
	int maxSize = 0;
	int minMajorAxis = 0;
	int maxMajorAxis = 0;
	int minMinorAxis = 0;
	int maxMinorAxis = 0;
	double minMajorMinorRatio = 1.0;
	double maxMajorMinorRatio = 10.0;
	
	double ellipseThr = 0;
	int[] areas;
	int[] vol;
	double[] sumX;
	double[] sumX2;
	double[] sumY;
	double[] sumY2;
	double[] sumXY;
	double[] sumZ;
	double[] sumZ2;
	double[] sumXZ;
	double[] sumYZ;
	short[] maxValInObj;
	int width;
	int height;
	int nSlices;
	boolean darkBkg = false;
	int minPeak;
	String outputFileName = "";
	
	//CONSTANTS
	public static final byte UNDEFINED = (byte)0;
	public static final byte NOT_OBJECT = (byte)1;
	public static final byte MAYBE_NOT_OBJECT = (byte)2;
	public static final byte OBJECT = (byte)3;
	
	public void run(String arg) 
	{
		if (showDialog())
			exec();
	}
	
	public boolean showDialog() 
	{
		GenericDialog gd = new GenericDialog("Per Object Ellipse fit");
		gd.addStringField("MinSize:", "1000", 4);
		gd.addStringField("MaxSize:", "50000", 4);
		gd.addStringField("EllipseThr:", "0.75", 4);
		gd.addStringField("MinMajorAxis:", "10", 4);
		gd.addStringField("MaxMajorAxis:", "30", 4);
		gd.addStringField("MinMinorAxis:", "5", 4);
		gd.addStringField("MaxMinorAxis:", "30", 4);
		gd.addStringField("MinMajorMinorRatio:", "1", 4);
		gd.addStringField("MaxMajorMinorRatio:", "3", 4);
		gd.addStringField("MinPeak:", "0", 4);
		gd.addCheckbox("DarkBkg", true);
		gd.addStringField("OutputFile:", "", 4);
		gd.showDialog();
		if (gd.wasCanceled())
			return false;
		minSize=Integer.valueOf(gd.getNextString()).intValue();
		maxSize=Integer.valueOf(gd.getNextString()).intValue();
		ellipseThr=Double.valueOf(gd.getNextString()).doubleValue();
		minMajorAxis=Integer.valueOf(gd.getNextString()).intValue();
		maxMajorAxis=Integer.valueOf(gd.getNextString()).intValue();
		minMinorAxis=Integer.valueOf(gd.getNextString()).intValue();
		maxMinorAxis=Integer.valueOf(gd.getNextString()).intValue();
		minMajorMinorRatio=Double.valueOf(gd.getNextString()).doubleValue();
		maxMajorMinorRatio=Double.valueOf(gd.getNextString()).doubleValue();
		minPeak=Integer.valueOf(gd.getNextString()).intValue();
		darkBkg = gd.getNextBoolean();
		outputFileName = gd.getNextString();
		return true;
	}

	public void exec() 
	{
		IJ.log("Start");
		long startTime0 = System.currentTimeMillis();
		if (IJ.versionLessThan("1.27t")) 
			return;   
		imp = WindowManager.getCurrentImage(); 
		stack=imp.getStack(); 
		width=imp.getWidth(); 
		height=imp.getHeight(); 
		nSlices=imp.getStackSize(); 
		int nPixels=width*height*nSlices; 
		String imageName = imp.getTitle(); 
		 
		//Create nodes 
        parNode=new int[nPixels]; 
         
		int type = imp.getType(); 
		int mask=0xff; 
		int slice=1; 
		int x,y,z,x0,x2,y0,y2,z0,z2; 
		int rest; 
		int i,j,k, t; 
		short p; 
		imArray = new short[nPixels]; 
		outputArray = new byte[nPixels];
		ellipseFit = new double[nPixels];
		majorAxis = new double[nPixels];
		minorAxis = new double[nPixels];
		sumX = new double[nPixels];
		sumX2 = new double[nPixels];
		sumY = new double[nPixels];
		sumY2 = new double[nPixels];
		sumXY = new double[nPixels];
		sumZ = new double[nPixels];
		sumZ2 = new double[nPixels];
		sumXZ = new double[nPixels];
		sumYZ = new double[nPixels]; 
		maxValInObj = new short[nPixels];
		if (type == ImagePlus.GRAY16)  
		{ 
			mask=0xffff; 
			i=0; 
			//Find min, max. Copy to imArray 
			int intP; 
			short[] pixels = (short[])stack.getPixels(1); 
			intP=(int)(mask&pixels[0]); 
			maxP=minP=(short)(intP/2); 
			pixels = null; 
			for(slice=1; slice<=nSlices; slice++) 
			{ 
				pixels = (short[])stack.getPixels(slice); 
				j=0; 
				for(y=0; y<height; y++) 
				{ 
					for(x=0; x<width; x++) 
					{ 
						intP=(int)(mask&pixels[j]);
						p = (short)(intP/2);
						if(p<minP)
							minP=p;
						if(p>maxP)
							maxP=p;
						imArray[i]=p;
						i++; 
						j++; 
					} 
				} 
				pixels = null; 
			} 
		}   
		else if (type == ImagePlus.GRAY8)  
		{ 
			mask=0xff; 
			i=0; 
			//Find min, max. Copy to imArray 
			byte[] pixels = (byte[])stack.getPixels(1); 
			maxP=minP=(short)(mask&pixels[0]); 
			pixels = null; 
			for(slice=1; slice<=nSlices; slice++) 
			{  
				pixels = (byte[])stack.getPixels(slice); 
				j=0; 
				for(y=0; y<height; y++) 
				{ 
					for(x=0; x<width; x++) 
					{ 
						p=(short)(mask&pixels[j]); 
						if(p<minP) 
							minP=p; 
						if(p>maxP) 
							maxP=p; 
						imArray[i]=p; 
						i++; 
						j++; 
					} 
				} 
				pixels = null; 
			} 
		} 
		else 
		{ 
			IJ.log("Pixel format not supported"); 
			return; 
		}
		IJ.log("nPix: " + nPixels + " MinP: " + minP + " MaxP: " + maxP);
		if (!darkBkg) 
		{ 
			for (i=0; i<nPixels; i++) 
			{ 
				imArray[i] = (short)((int)minP + (int)maxP - (int)imArray[i]); 
			} 
		} 	 
		areas = new int[nPixels]; 
		vol = new int[nPixels]; 
		for (i=0; i<nPixels; i++) 
		{ 
			areas[i]=1;
			vol[i]=1;
			maxValInObj[i] = imArray[i];
		}
		i=0;
		for(slice=1; slice<=nSlices; slice++) 
		{ 
			for(y=0; y<height; y++)
			{
				for(x=0; x<width; x++)
				{
					z = slice - 1;
					sumX[i] = x;
					sumX2[i] = x*x;
					sumY[i] = y;
					sumY2[i] = y*y;
					sumXY[i] = x*y;
					sumZ[i] = z;
					sumZ2[i] = z*z;
					sumXZ[i] = x*z;
					sumYZ[i] = y*z;
					i++;
				}
			} 
		}
		 
		//Sort points 
		// create a counting array, counts, with a member for  
		// each possible discrete value in the input.   
		// initialize all counts to 0. 
		int nLevels = maxP-minP + 1; 
		int[] counts = new int[nLevels]; 
		double[] grayHist = new double[nLevels];  
		// for each value in the unsorted array, increment the 
		// count in the corresponding element of the count array 
		for (i=0; i<nPixels; i++) 
		{ 
			counts[imArray[i]-minP]++; 
		}  
		// accumulate the counts - the result is that counts will hold 
		// the offset into the sorted array for the value associated with that index 
		for (i=1; i<nLevels; i++) 
		{  
			counts[i] += counts[i-1]; 
		} 
		// store the elements in a new ordered array 
		int[] sortedIndex = new int[nPixels]; 
		for (i = nPixels-1; i >= 0; i--) 
		{ 
			// decrementing the counts value ensures duplicate values in A 
			// are stored at different indices in sorted. 
			sortedIndex[--counts[imArray[i]-minP]] = i;                 
		} 
 
		//Init nodes 
		for (i=0; i<nPixels; i++) 
		{ 
			parNode[i]=i; 
		} 
		//Search in decreasing order 
		int curNode; 
		int adjNode; 
		for (i = nPixels-1; i >= 0; i--) 
		{ 
			j=sortedIndex[i]; 
			curNode=j; 
			 
			z=j/(width*height); 
			rest=j-z*(width*height); 
			y=rest/width; 
			x=rest-y*width; 
			 
			z0=z-1; 
			z2=z+1; 
			y0=y-1; 
			y2=y+1; 
			x0=x-1; 
			x2=x+1; 
 
			//Later neigbours x2,y2,z2 
			boolean foundNeighbour = false; 
			if(z2<nSlices) 
			{ 
				k=x+width*(y+z2*height); 
				if(imArray[k]>=imArray[j]) 
				{ 
					adjNode=findNode(k); 
					if(curNode!=adjNode) 
					{ 
						curNode=mergeNodes(adjNode,curNode); 
					} 
				} 
			} 
			if(y2<height) 
			{ 
				k=x+width*(y2+z*height); 
				if(imArray[k]>=imArray[j]) 
				{ 
					adjNode=findNode(k); 
					if(curNode!=adjNode) 
					{ 
						curNode=mergeNodes(adjNode,curNode); 
					} 
				} 
			} 
			if(x2<width) 
			{ 
				k=x2+width*(y+z*height); 
				if(imArray[k]>=imArray[j]) 
				{ 
					adjNode=findNode(k); 
					if(curNode!=adjNode) 
					{ 
						curNode=mergeNodes(adjNode,curNode); 
					} 
				} 
			} 
			//Earlier neighbours x0,y0,z0. No need to check = 
			if(x0>=0) 
			{ 
				k=x0+width*(y+z*height); 
				if(imArray[k]>imArray[j]) 
				{ 
					adjNode=findNode(k); 
					if(curNode!=adjNode) 
					{ 
						curNode=mergeNodes(adjNode,curNode); 
					} 
					 
				} 
			} 
			if(y0>=0) 
			{ 
				k=x+width*(y0+z*height); 
				if(imArray[k]>imArray[j]) 
				{ 
					adjNode=findNode(k); 
					if(curNode!=adjNode) 
					{ 
						curNode=mergeNodes(adjNode,curNode); 
					} 
				} 
			} 
			if(z0>=0) 
			{ 
				k=x+width*(y+z0*height); 
				if(imArray[k]>imArray[j]) 
				{ 
					adjNode=findNode(k); 
					if(curNode!=adjNode) 
					{ 
						curNode=mergeNodes(adjNode,curNode); 
					} 
				} 
			} 
		}
		for (i = nPixels-1; i >= 0; i--)
		{
			j=sortedIndex[i];
			if (outputArray[j] == UNDEFINED)
			{
				int e = j;
				while(imArray[e] == imArray[parNode[e]] && outputArray[e] == UNDEFINED)
					e = parNode[e];
				if (outputArray[e] == UNDEFINED)
				{
					findBestEllipseLevel(j, minSize, maxSize, ellipseThr, minMajorAxis, maxMajorAxis, minMinorAxis, maxMinorAxis, minMajorMinorRatio, maxMajorMinorRatio, minPeak, -1);
				}
				else
				{
					int e1 = j;
					while(e1 != e)
					{
						outputArray[e1] = outputArray[e];
						e1 = parNode[e1];
					}
				}
			}
		}
		//Handle MAYBE
		for (i = nPixels-1; i >= 0; i--)
		{
			j=sortedIndex[i];
			if (outputArray[j] == MAYBE_NOT_OBJECT)
			{
				findBestEllipseLevel(j, minSize, maxSize, ellipseThr, minMajorAxis, maxMajorAxis, minMinorAxis, maxMinorAxis, minMajorMinorRatio, maxMajorMinorRatio, minPeak, -1);
			}
		}
		for (i = nPixels-1; i >= 0; i--)
		{
			j=sortedIndex[i];
			if (outputArray[j] == MAYBE_NOT_OBJECT)
			{
				findObjBelow(j, maxSize);
			}
		}
		
		i=0;  
		if (outputFileName.length() > 0)
		{
			PrintStream myOutput = null;
			try
			{
				String f = outputFileName.replaceAll("%20", " ");
				File file = new File(f);
				boolean fileExisted = file.isFile();
				FileOutputStream fout = new FileOutputStream(file, true);
				myOutput = new PrintStream(fout);
				if (!fileExisted)
				{
					myOutput.println("x0;y0;z0;x;y;z;vol");
				}
				if (type == ImagePlus.GRAY16)  
				{ 
					for( slice=1; slice<=nSlices; slice++) 
					{ 
						short[] pixels = (short[])stack.getPixels(slice); 
						j=0; 
						for( y=0; y<height; y++) 
						{ 
							for( x=0; x<width; x++) 
							{ 
								if(outputArray[i] == OBJECT)
									t=255;
								else
									t=0;
								if (outputArray[i] == OBJECT && outputArray[parNode[i]] != OBJECT)
									myOutput.println(x + ";" + y + ";" + (slice-1) + ";" + (sumX[i]/vol[i]) + ";" + (sumY[i]/vol[i]) + ";" + (sumZ[i]/vol[i]) + ";" + vol[i]);
								pixels[j]=(short)t; 
								i++; 
								j++; 
							} 
						} 
						pixels = null; 
					} 
				}
				else if (type == ImagePlus.GRAY8)  
				{ 
					for( slice=1; slice<=nSlices; slice++) 
					{ 
						byte[] pixels = (byte[])stack.getPixels(slice); 
						j=0; 
						for( y=0; y<height; y++) 
						{ 
							for( x=0; x<width; x++) 
							{ 
								if(outputArray[i] == OBJECT) 
									t=255; 
								else 
									t=0; 
								if (outputArray[i] == OBJECT && outputArray[parNode[i]] != OBJECT)
									myOutput.println(x + ";" + y + ";" + (slice-1) + ";" + (sumX[i]/vol[i]) + ";" + (sumY[i]/vol[i]) + ";" + (sumZ[i]/vol[i]) + ";" + vol[i]);
								pixels[j]=(byte)t; 
								i++; 
								j++; 
							} 
						} 
						pixels = null; 
					} 
				}
			}
			catch (IOException e) 
			{ 
				IJ.log("Error write file: ".concat(e.getMessage())); 
			}
			finally 
			{
				if (myOutput != null) 
					myOutput.close();
			}
		}
		else
		{
			if (type == ImagePlus.GRAY16)  
			{ 
				for( slice=1; slice<=nSlices; slice++) 
				{ 
					short[] pixels = (short[])stack.getPixels(slice); 
					j=0; 
					for( y=0; y<height; y++) 
					{ 
						for( x=0; x<width; x++) 
						{ 
							if(outputArray[i] == OBJECT)
								t=255;
							else
								t=0;
							pixels[j]=(short)t; 
							i++; 
							j++; 
						} 
					} 
					pixels = null; 
				} 
			}
			else if (type == ImagePlus.GRAY8)  
			{ 
				for( slice=1; slice<=nSlices; slice++) 
				{ 
					byte[] pixels = (byte[])stack.getPixels(slice); 
					j=0; 
					for( y=0; y<height; y++) 
					{ 
						for( x=0; x<width; x++) 
						{ 
							if(outputArray[i] == OBJECT) 
								t=255; 
							else 
								t=0; 
							pixels[j]=(byte)t; 
							i++; 
							j++; 
						} 
					} 
					pixels = null; 
				} 
			}
		}
		if(type!=ImagePlus.GRAY8) 
		{ 
			//Convert to 8-bit 
			ImageStack stack2 = new ImageStack(width, height); 
			String stackLabel; 
			ImageProcessor ip1, ip2; 
			boolean scale = false; 
			for( slice=1; slice<=nSlices; slice++)  
			{ 
				stackLabel = stack.getSliceLabel(1); 
				ip1 = stack.getProcessor(1); 
				ip2 = ip1.convertToByte(scale); 
				stack.deleteSlice(1); 
				stack2.addSlice(stackLabel, ip2); 
			} 
			imp.setStack(null, stack2); 
			stack2 = null; 
		} 
		imp.updateAndRepaintWindow();
	 
		//Deallocate arrays	
		parNode = null; 
		imArray = null; 
		areas = null;
		vol = null;
		counts = null; 
		sortedIndex = null; 
		outputArray = null; 
		ellipseFit = null; 
		majorAxis = null; 
		minorAxis = null; 
		sumX = null; 
		sumX2 = null; 
		sumY = null; 
		sumY2 = null; 
		sumXY = null; 
		sumZ = null; 
		sumZ2 = null; 
		sumXZ = null; 
		sumYZ = null; 
		maxValInObj = null;
		imp = null; 
		stack = null; 
		IJ.freeMemory(); 
		IJ.log("Done");
		long stopTime = System.currentTimeMillis();
		IJ.log("TotalTime " + (stopTime - startTime0) + " ms");
	}
	
	public int findNode(int e)
	{
		if(parNode[e]!=e)
		{
			int root = findNode(parNode[e]);
			//parNode[e] = root; //This cannot be used here
			return root;
		}
		else
		{
			return e;
		}
	}
	
	public boolean findObjBelow(int e, double maxS)
	{
		int y=e/width;
		int x=e-y*width;
		if (parNode[e] == e)
		{
			outputArray[e] = NOT_OBJECT;			
			return false;
		}
		if (outputArray[parNode[e]] == OBJECT)
		{
			outputArray[e] = OBJECT;
			return true;
		}
		if (areas[parNode[e]] > maxS)
		{	
			outputArray[e] = NOT_OBJECT;			
			return false;
		}
		boolean found = findObjBelow(parNode[e], maxS);
		if (found)
			outputArray[e] = outputArray[parNode[e]];
		else
			outputArray[e] = NOT_OBJECT;
		return found;			
	}
	
	public int findBestEllipseLevel(int e, double minS, double maxS, double eThr, double minMajor, double maxMajor, double minMinor, double maxMinor, double minMajorMinor, double maxMajorMinor, int minPeak, int optE)
	{
		double optEf;
		double optArea;
		if (optE >= 0)
		{
			optEf = ellipseFit[optE];
			optArea = areas[optE];
		}
		else
		{
			optEf = 0;
			optArea = 0;
		}
		int startE = e;
		while(imArray[e] == imArray[parNode[e]] && parNode[e] != e)
			e = parNode[e];
		if (outputArray[e] == OBJECT)
		{
			return e;
		}
		if (outputArray[e] == NOT_OBJECT)
		{
			return optE;
		}
		if (parNode[e] == e)
		{
			outputArray[e] = NOT_OBJECT;
			return optE;
		}
		if (areas[e] > maxS)
		{
			outputArray[startE] = NOT_OBJECT;
			while(startE != parNode[startE] && outputArray[parNode[startE]] == UNDEFINED)
			{
				startE = parNode[startE];
				outputArray[startE] = NOT_OBJECT;
			}
			return optE;
		}
		double majorMinorRatio = majorAxis[e]/minorAxis[e];
		short diff = (short)(maxValInObj[e] - imArray[parNode[e]]);
		if (areas[e] >= minS && ellipseFit[e] > optEf && ellipseFit[e] > eThr && majorAxis[e] >= minMajor && majorAxis[e] <= maxMajor && minorAxis[e] >= minMinor && minorAxis[e] <= maxMinor && majorMinorRatio >= minMajorMinor && majorMinorRatio <= maxMajorMinor && diff >= minPeak)
		{			
			optEf = ellipseFit[e];
			optArea = areas[e];
			optE = e;
		}
		optE = findBestEllipseLevel(parNode[e], minS, maxS, eThr, minMajor, maxMajor, minMinor, maxMinor, minMajorMinor, maxMajorMinor, minPeak, optE);
		if (optE >= 0)
		{
			optEf = ellipseFit[optE];
			optArea = areas[optE];
		}
		else
		{
			optEf = 0;
			optArea = 0;
		}
		if (optE >= 0 && imArray[e]  >= imArray[optE]) //Opt found below
		{
			// IJ.log("e " + e + " optL " + imArray[optE] + " area " + areas[optE] + " ef " + ellipseFit[optE]);
			while(startE != e)
			{
				outputArray[startE] = OBJECT;
				startE = parNode[startE];
			}
			outputArray[e] = OBJECT;
			return optE;
		}
		
		// Below best level
		outputArray[startE] = MAYBE_NOT_OBJECT;
		if (optArea > areas[startE])
				areas[startE] = 0;
		else
			areas[startE] -= optArea;
		
		while(startE != e)
		{
			startE = parNode[startE];
			outputArray[startE] = MAYBE_NOT_OBJECT;			
			if (optArea > areas[startE])
				areas[startE] = 0;
			else
				areas[startE] -= optArea;
		}
		return optE;	
	}
	
	public int mergeNodes(int e1,int e2)
	{
		int res;
		int m;
		
		if(imArray[e1]==imArray[e2])
		{
			res=Math.max(e1,e2);
			m=Math.min(e1,e2);
		}
		else
		{
			res=e2;
			m=e1;
		}
		areas[res] += areas[m];
		vol[res] = areas[res];
		parNode[m]=res;
		sumX[res] += sumX[m];
		sumX2[res] += sumX2[m];
		sumY[res] += sumY[m];
		sumY2[res] += sumY2[m];
		sumXY[res] += sumXY[m];
		sumZ[res] += sumZ[m];
		sumZ2[res] += sumZ2[m];
		sumXZ[res] += sumXZ[m];
		sumYZ[res] += sumYZ[m];
		if (maxValInObj[m] > maxValInObj[res])
			maxValInObj[res] = maxValInObj[m];
		if (areas[res] > 1)
		{
			double varX = (sumX2[res] - sumX[res]*sumX[res]/areas[res]);///(areas[res]-1);
			double varY = (sumY2[res] - sumY[res]*sumY[res]/areas[res]);///(areas[res]-1);
			double varZ = (sumZ2[res] - sumZ[res]*sumZ[res]/areas[res]);///(areas[res]-1);
			double covXY = (sumXY[res] - sumX[res]*sumY[res]/areas[res]);///(areas[res]-1);
			double covXZ = (sumXZ[res] - sumX[res]*sumZ[res]/areas[res]);///(areas[res]-1);
			double covYZ = (sumYZ[res] - sumY[res]*sumZ[res]/areas[res]);///(areas[res]-1);
			double lambda1 = 0;
			double lambda2 = 0;
			double lambda3 = 0;
			
			double p1 = covXY*covXY + covXZ*covXZ + covYZ*covYZ;
			// IJ.log("res " + res + " v " + imArray[res] + " area " + areas[res] + " p1 " + p1 + " sumX " + sumX[res] + " sumY " + sumY[res] + " sumZ " + sumZ[res] + " sumX2 " + sumX2[res] + " sumY2 " + sumY2[res] + " sumZ2 " + sumZ2[res] + " sumXY " + sumXY[res] + " sumXZ " + sumXZ[res] + " sumYZ " + sumYZ[res]);
			// IJ.log("varX " + varX + " varY " + varY + " varZ " + varZ + " covXY " + covXY + " covXZ " + covXZ + " covYZ " + covYZ + " p1 " + p1);
			
			if (p1 == 0)
			{
				lambda1 = varX;
				lambda2 = varY;
				lambda3 = varZ;
			}
			else
			{
				double q = (varX + varY + varZ)/3.0;
				double p2 = (varX - q) * (varX - q) + (varY - q) * (varY - q) + (varZ - q) * (varZ - q) + 2 * p1;
				double p0 = Math.sqrt(p2/6.0);
				double b00 = (varX-q) / p0;
				double b01 = covXY / p0;
				double b02 = covXZ / p0;
				double b11 = (varY-q) / p0;
				double b12 = covYZ / p0;
				double b22 = (varZ-q) / p0;
				double detB = b00*(b11*b22-b12*b12) + b01*(b02*b12-b01*b22) + b02*(b01*b12-b11*b02);
				double r = detB / 2.0;
				double phi;
				if (r <= -1)
					phi = Math.PI / 3.0;
				else if (r >= 1)
					phi = 0;
				else
					phi = Math.acos(r) / 3.0;
				
				lambda1 = q + 2 * p0 * Math.cos(phi);
				lambda3 = q + 2 * p0 * Math.cos(phi + (2 * Math.PI/3.0));
				lambda2 = 3 * q - lambda1 - lambda3;
			}
			// double l1 = 2 * Math.sqrt(lambda1/areas[res]);
			// double l2 = 2 * Math.sqrt(lambda2/areas[res]);
			// double l3 = 2 * Math.sqrt(lambda3/areas[res]);
			double l1 = Math.sqrt(5.0*lambda1/areas[res]); //Don't know why 5.0 should be in there but it gives correct result!!!
			double l2 = Math.sqrt(5.0*lambda2/areas[res]);
			double l3 = Math.sqrt(5.0*lambda3/areas[res]);
			double ellipsoidVolume = 4.0 * Math.PI * l1 * l2 * l3 / 3.0;
			if (ellipsoidVolume > 0)
			{
				ellipseFit[res] = areas[res]/ellipsoidVolume;
				majorAxis[res] = 2.0 * l1;
				minorAxis[res] = 2.0 * l3;
				// IJ.log ("v " + imArray[res] + " lambda1 " + lambda1 + " lambda2 " + lambda2 + " lambda3 " + lambda3 + " ellipsoidVolume " + ellipsoidVolume + " volume " + areas[res] + " ellipseFit " + ellipseFit[res] + " l1 " + l1 + " l2 " + l2 + " l3 " + l3);
			}
		}
		return res;
	}
	
	void error() 
	{
		IJ.showMessage("PerObjectEllipsefit3D", "Error");
	}
}
